import itertools
from collections import OrderedDict
from itertools import product
import os
import requests
import json

# MATRX stuff
from matrx import WorldBuilder, utils
import numpy as np
from matrx.actions import MoveNorth, OpenDoorAction, CloseDoorAction
from matrx.actions.move_actions import MoveEast, MoveSouth, MoveWest
from matrx.grid_world import GridWorld, DropObject, GrabObject, AgentBody
from matrx.objects import EnvObject, SquareBlock
from matrx.world_builder import RandomProperty
from matrx.goals import WorldGoal, CollectionGoal
from moving_out.case.gui import visualization_server
from matrx.goals import LimitedTimeGoal

# custom code
from moving_out.agents.non_faulty_agent import NonFaultyAgent, non_faulty_build_boxes, world_height, world_width
from moving_out.agents.faulty_agent import FaultyAgent, faulty_build_boxes
from moving_out.agents.tutorial_agent import TutorialAgent, tutorial_build_boxes
from moving_out.agents.human_agent import CustomHumanAgent
from moving_out.actions.custom_actions import LowerObject, CarryObject


# Some general settings
tick_duration = 1/100  # 0.1s or lower tick duration recommended, to keep the human agent responsive
random_seed = 1
verbose = False
key_action_map = {  # Controls for the human agent
    'w': MoveNorth.__name__,
    'd': MoveEast.__name__,
    's': MoveSouth.__name__,
    'a': MoveWest.__name__,
    'p': LowerObject.__name__,
    'l': CarryObject.__name__
}




def create_builder(scenario):

    # Set numpy's random generator
    np.random.seed(random_seed)

    # The world size
    world_size = [world_width, world_height]

    # Create our world builder
    goal = LimitedTimeGoal(max_nr_ticks=10000000000000000000)
    builder = WorldBuilder(shape=world_size, tick_duration=tick_duration, random_seed=random_seed, run_matrx_api=True,
                           run_matrx_visualizer=False, verbose=verbose, visualization_bg_clr="#f0f0f0",
                           visualization_bg_img="", simulation_goal=goal)

    # Create rooms
    builder.add_room(top_left_location=[0,1], width=9, height=11, name="roomHuman", door_locations=[(8,5)], doors_open=True)
    builder.add_room(top_left_location=[16, 1], width=9, height=11, name="roomRobot", door_locations=[(16,5)], doors_open=True)

    # Create dropzone
    builder.add_area(top_left_location=[0,0], width=world_width, height=1, name="truck", is_drop_zone=True, visualize_opacity=0.5)

    # Create safe zone
    builder.add_area(top_left_location=[10, 3], width=5, height=3, name="safe", is_drop_zone=True, visualize_colour="#f7c281", visualize_opacity=0.5)

    # check which robot to place and which boxes to put down
    build_boxes = []
    robotbrain = None
    if scenario == 0:
        robotbrain = TutorialAgent()
        build_boxes = tutorial_build_boxes
    elif scenario == 1:
        robotbrain = NonFaultyAgent()
        build_boxes = non_faulty_build_boxes
    elif scenario == 2:
        robotbrain = NonFaultyAgent()
        build_boxes = non_faulty_build_boxes

    # Create boxes and ghost boxes
    for index, box in enumerate(build_boxes):
        if box["owner"] == "human":
            if box["weight"] == "light":
                img_name = "light_box_human.png"
            else: img_name = "heavy_box_human.png"
        else:
            if box["weight"] == "light":
                img_name = "light_box_robot.png"
            else: img_name = "heavy_box_robot.png"
        builder.add_object(location=box["location"], name="box", is_traversable=True, is_movable=True, img_name=img_name,
                           weight=box["weight"], owner=box["owner"], number=index, safezone = False, visualize_size=0.8)
        builder.add_object(location=(index, 0), name="ghostbox", is_traversable=True, is_movable=False,
                           img_name=img_name,
                           weight=box["weight"], owner=box["owner"], number=index, visualize_opacity=0.4, visualize_size=0.8)

    # Custom human agent
    humanbrain = CustomHumanAgent(max_carry_objects=1)
    builder.add_human_agent([1, 3], humanbrain, name="human", key_action_map=key_action_map, is_traversable=True,
                            img_name="human.png", customizable_properties=['score', 'img_name'], score=0)

    # Custom artificial agent
    # Make it not possible to move diagonally so both agents have same capabilities
    builder.add_agent([7, 10], robotbrain, name="robot", is_traversable=True, img_name="robot.png",
                      possible_actions=[MoveEast.__name__, MoveSouth.__name__, MoveWest.__name__, MoveNorth.__name__,
                                        LowerObject.__name__, CarryObject.__name__],
                      customizable_properties=['score', 'img_name'], score=0)
    # Return the builder
    return builder


def run(scenario):
    # By creating scripts that return a builder, we can define infinite number of use cases and select them (in the
    # future) through a UI.
    builder = create_builder(scenario)
    # we set a path pointing to our custom images
    media_folder = os.path.join(os.path.dirname(os.path.realpath(__file__)), "..", "..", "images")

    # startup world-overarching MATRX scripts, such as the api and/or visualizer if requested
    builder.startup(media_folder=media_folder)

    print("Starting custom visualizer")
    visualizer = visualization_server.run_matrx_visualizer(verbose=False, media_folder=media_folder, s=scenario)

    for world in builder.worlds():
        world.run(builder.api_info)

    r = requests.get("http://localhost:") + str(visualization_server.port) + "/shutdown_visualizer"
    visualizer.join()

    builder.stop()